/*
  ES100 ADK Tech Demo V2 (CANADUINO)
  Demo firmware for the ES100 WWVB-BPSK ADK with DS1307 RTC + 20x4 LCD.
  Target: Arduino UNO / ATmega328P
  Works with the CANADUINO ES100 Application Development Kit (ADK) V1 and V2.

  ES100 connections:
    IRQ -> D2 (INT0)   (ADK has pull-up on IRQ)
    EN  -> D13         (ADK has 10k pull-down on EN; driven by MCU through level shifter)

  LCD (parallel 20x4, LiquidCrystal 4-bit):
    RS=D4, EN=D5, D4=D8, D5=D9, D6=D10, D7=D11

  Buttons (active-low, 1k series to GND; enable INPUT_PULLUP in MCU):
    S1 -> D3   (Prev / Up)
    S2 -> D6   (Menu / Enter)
    S3 -> D7   (Next / Down)

  Accessories:
    A0..A3 available. One screen shows raw ADC readings.

  Notes:
    - I2C clock set to 100 kHz for better DS1307 compatibility.
    - ES100 library handles timezone + optional DST. RTC stores LOCAL time (not UTC).
    - Serial output always on (9600 baud by default).
*/

#include <Wire.h>
#include <LiquidCrystal.h>
#include <DS1307RTC.h>
#include <TimeLib.h>
#include <EEPROM.h>
#include <stddef.h>

#include "ES100.h"

// ===================== HARDWARE PINS =====================
/*
  HARDWARE OVERVIEW
  -----------------
  This ADK is built around an ATmega328P (Arduino UNO compatible) and an Everset ES100
  WWVB-BPSK receiver.

  Signals (ADK wiring)
  - ES100 IRQ -> D2 (INT0). The ES100 asserts IRQ when a new decode (or status) is ready.
  - ES100 EN  -> D13 (via level shifter). EN is normally held LOW (10k to GND) to keep the
    ES100 disabled until the sketch enables it.
  - LCD 20x4  -> classic HD44780-compatible parallel interface (LiquidCrystal 4-bit mode).
  - Buttons   -> S1=D3, S2=D6, S3=D7. Buttons pull the pin to GND when pressed, so we enable
    the internal pullups (INPUT_PULLUP) and treat "pressed" as LOW.

  Notes
  - I²C bus speed is kept at 100 kHz for DS1307 compatibility.
  - Serial output is always enabled and mirrors important status events from the LCD.
*/
static const uint8_t PIN_ES100_IRQ = 2;   // INT0
static const uint8_t PIN_ES100_EN  = 13;

static const uint8_t PIN_S1 = 3;
static const uint8_t PIN_S2 = 6;
static const uint8_t PIN_S3 = 7;

// LCD pins (ADK wiring)
static const uint8_t LCD_RS = 4;
static const uint8_t LCD_EN = 5;
static const uint8_t LCD_D4 = 8;
static const uint8_t LCD_D5 = 9;
static const uint8_t LCD_D6 = 10;
static const uint8_t LCD_D7 = 11;

// ===================== DEFAULT SETTINGS =====================
/*
  DEFAULT BEHAVIOR / CUSTOMER SETTINGS
  -----------------------------------
  The ADK is meant to be experimented with. Therefore most operational parameters can be changed
  at runtime from the on-device menu and are stored in EEPROM.

  The defaults below are used:
  - when EEPROM does not contain valid settings yet, or
  - when the EEPROM checksum does not match.

  You can still change these constants to define "factory defaults", but customers can override them
  via the CONFIG menu without recompiling.
*/
#define DEF_ZONE_HOURS       (-5)   // Time Zone UTC (-5 equals EST)
#define DEF_DST_AUTO         (1)    // 1=AUTO, 0=OFF
#define DEF_ANT_MODE         (ES100_ANTMODE_AUTO)

#define DEF_SYNC_OK_MIN      (15)   // resync interval after success
#define DEF_SYNC_FAIL_MIN    (2)    // retry interval after failure

// RX mode: 0=FULL only, 1=FULL until first OK then TRACK, 2=TRACK only
#define DEF_RX_MODE          (1)

#define SERIAL_BAUD          (9600)
#define UI_REFRESH_MS        (200)
#define DEBOUNCE_MS          (30)
#define LONGPRESS_MS         (900)

// ===================== TYPES (keep above any function prototypes) =====================
/*
  TYPES
  -----
  Arduino IDE 2.x auto-generates function prototypes. To avoid compile issues, all custom types
  are declared before any function definitions that might reference them.

  Settings:
    - Stored in EEPROM with a magic value and a simple checksum.
    - Values are intentionally compact (bytes) to keep EEPROM usage small and stable.

  Buttons:
    - Simple debouncing + short/long press detection.
    - Long press is used for "force RX now" and for "save+exit" inside the menu.
*/
enum Screen : uint8_t {
  SCR_CLOCK = 0,
  SCR_RX    = 1,
  SCR_ES100 = 2,
  SCR_ANALOG= 3,
  SCR_CFG   = 4,
  SCR_COUNT
};

enum BtnEvent : uint8_t { EV_NONE=0, EV_SHORT=1, EV_LONG=2 };

struct BtnState {
  uint8_t pin;
  bool stableHigh;              // true = HIGH (not pressed), false = LOW (pressed)
  unsigned long lastChangeMs;
  unsigned long pressedAtMs;
  bool longSent;
};

struct Settings {
  uint16_t magic;     // 0xE510
  int8_t   zoneHours; // -12..+14 typical
  uint8_t  dstAuto;   // 0=OFF, 1=AUTO
  uint8_t  antMode;   // ES100AntennaMode (0..3)
  uint8_t  okMinutes; // 1..120
  uint8_t  failMinutes;//1..60
  uint8_t  rxMode;    // 0..2
  uint16_t checksum;  // sum of bytes [0..offsetof(checksum)-1]
};

// ===================== FORWARD DECLARATIONS =====================
/*
  FORWARD DECLARATIONS
  --------------------
  These are declared explicitly so the build is deterministic across Arduino IDE versions.
  (The Arduino build system can generate prototypes automatically, but explicit declarations
  avoid surprises.)
*/
static uint16_t checksum16(const uint8_t* p, size_t n);
static bool cfgIsValid(const Settings& s);
static void cfgSetDefaults(Settings& s);
static void cfgLoad(Settings& s);
static void cfgSave(const Settings& s);

static void applyCfgToEs100();
static void scheduleNext(bool lastWasOk);/*
  Start an ES100 reception attempt.

  trackingMode = false:
    Run a full 1-minute frame reception (more robust, slower).

  trackingMode = true:
    Run a tracking reception (faster updates once a strong signal is established).

  The ES100 decides which antenna to use based on its internal logic plus our antenna policy.
*/


static void startReception(bool tracking);
static void handleIrqIfAny();

static BtnEvent pollButton(BtnState& b);
static void lcdLine(uint8_t row, const char* s);

static const char* antModeStr(uint8_t m);
static bool esTimeSane(const ES100DateTime& t) {
  if (t.month < 1 || t.month > 12) return false;
  if (t.day   < 1 || t.day   > 31) return false;
  if (t.hour  > 23) return false;
  if (t.minute > 59) return false;
  if (t.second > 59) return false;

  // ES100 gives a 2-digit year (00..99). Reject obviously bogus years like 99 (=2099).
  // Accept 2000..2079 by default (adjust if you need later years).
  if (t.year >= 80) return false;

  return true;
}

static const char* dstStateStr(uint8_t st);
static void fmtUptime(char* out, unsigned long ms);

static void renderScreen();
static void renderClock();
static void renderRx();
static void renderEs100();
static void renderAnalog();/*
  Render the CONFIG screen.

  - NAV mode: S1/S3 moves between items
  - EDIT mode: S1/S3 changes the value of the selected item
  - S2 toggles NAV <-> EDIT
  - S2 hold saves (if changed) and exits
*/


static void renderCfg();

// ===================== OBJECTS / GLOBALS =====================
/*
  GLOBAL STATE
  ------------
  - es100:     ES100 library instance (I²C + IRQ driven)
  - rtc:       DS1307 clock (local time is stored in the RTC)
  - lcd:       20x4 character LCD via LiquidCrystal
  - cfg:       current settings loaded from EEPROM (or defaults)
  - screen:    which UI screen is currently shown
  - receiving: true while an ES100 reception attempt is running
  - okCount / failCount: success/failure counters since boot
  - last...:   most recent ES100 status snapshot and timestamps for user feedback
*/
LiquidCrystal lcd(LCD_RS, LCD_EN, LCD_D4, LCD_D5, LCD_D6, LCD_D7);
ES100 es100;

static volatile uint16_t g_irqCount = 0;

static Settings cfg;

static Screen g_screen = SCR_CLOCK;
static unsigned long g_lastUiMs = 0;

static bool receiving = false;
static unsigned long nextRxDueMs = 0;

static uint32_t okCount = 0;
static uint32_t failCount = 0;
static unsigned long lastOkMs = 0;
static unsigned long lastAttemptMs = 0;

// last ES100 observed
static ES100Status0 lastStatus {};
static ES100NextDst  lastNextDst {};
static ES100DateTime lastEsTime {};
static uint8_t lastIrqStatus = 0;
static uint8_t lastDeviceId = 0;

// config UI state
static bool cfgEditing = false;
static uint8_t cfgIndex = 0;
static bool cfgDirty = false;

// buttons
static BtnState b1{PIN_S1, true, 0, 0, false};
static BtnState b2{PIN_S2, true, 0, 0, false};
static BtnState b3{PIN_S3, true, 0, 0, false};

// ===================== IRQ ISR =====================
/*
  ES100 IRQ HANDLING
  ------------------
  The ES100 signals the MCU with an interrupt whenever new data is available.
  The ISR intentionally does as little as possible:
    - it records that an IRQ happened,
    - it snapshots a timestamp (millis()),
  and then the main loop reads the ES100 registers over I²C.

  This keeps ISR execution short and avoids doing I²C work inside an interrupt context.
*/
static void isr_es100_irq() {
  g_irqCount++;
}

// ===================== EEPROM / SETTINGS =====================
/*
  EEPROM SETTINGS STORAGE
  -----------------------
  We store a small struct in EEPROM:
    - magic field (to detect first boot / uninitialized EEPROM)
    - checksum (to detect corruption)
    - the actual configurable parameters

  The CONFIG screen lets the user:
    - browse settings with S1/S3
    - enter EDIT mode with S2
    - change values with S1/S3
    - hold S2 to save+exit (or exit without saving)
*/
static uint16_t checksum16(const uint8_t* p, size_t n) {
  uint32_t s = 0;
  for (size_t i = 0; i < n; i++) s += p[i];
  return (uint16_t)(s & 0xFFFF);
}

static bool cfgIsValid(const Settings& s) {
  if (s.magic != 0xE510) return false;
  uint16_t c = checksum16((const uint8_t*)&s, offsetof(Settings, checksum));
  return (c == s.checksum);
}

static void cfgSetDefaults(Settings& s) {
  s.magic       = 0xE510;
  s.zoneHours   = (int8_t)DEF_ZONE_HOURS;
  s.dstAuto     = (uint8_t)DEF_DST_AUTO;
  s.antMode     = (uint8_t)DEF_ANT_MODE;
  s.okMinutes   = (uint8_t)DEF_SYNC_OK_MIN;
  s.failMinutes = (uint8_t)DEF_SYNC_FAIL_MIN;
  s.rxMode      = (uint8_t)DEF_RX_MODE;
  s.checksum    = 0;
  s.checksum    = checksum16((const uint8_t*)&s, offsetof(Settings, checksum));
}

static void cfgLoad(Settings& s) {
  EEPROM.get(0, s);
  if (!cfgIsValid(s)) {
    cfgSetDefaults(s);
    EEPROM.put(0, s);
  }
}

static void cfgSave(const Settings& s) {
  Settings out = s;
  out.checksum = checksum16((const uint8_t*)&out, offsetof(Settings, checksum));
  EEPROM.put(0, out);
}

// ===================== HELPERS =====================
/*
  HELPER FUNCTIONS
  ----------------
  - Time formatting helpers: show "time since last sync" in a readable way.
  - Button helpers: debounce and generate events (short/long press).
  - ES100 helpers: translate status bits into text and decide which RX mode to run next.
*/
static void applyCfgToEs100() {
  es100.timezone = cfg.zoneHours;
  es100.DSTenabled = (cfg.dstAuto != 0);
  es100.antennaMode = (ES100AntennaMode)cfg.antMode;
}

static void scheduleNext(bool lastWasOk) {
  unsigned long now = millis();
  uint8_t mins = lastWasOk ? cfg.okMinutes : cfg.failMinutes;
  nextRxDueMs = now + (unsigned long)mins * 60000UL;
}

static void startReception(bool tracking) {
  receiving = true;
  lastAttemptMs = millis();
  g_irqCount = 0;

  es100.startRx(tracking ? 1 : 0);

  Serial.print(F("ES100 RX start: "));
  Serial.println(tracking ? F("TRACK") : F("FULL"));
}

static void handleIrqIfAny() {
  if (!receiving) return;
  if (g_irqCount == 0) return;

  // consume IRQ(s)
  g_irqCount = 0;

  lastIrqStatus = es100.getIRQStatus();
  lastStatus    = es100.getStatus0();

  bool ok = (lastStatus.rxOk != 0);

  Serial.print(F("IRQ=0x")); Serial.print(lastIrqStatus, HEX);
  Serial.print(F(" RXok=")); Serial.print(ok ? 1 : 0);
  Serial.print(F(" ant=")); Serial.print(lastStatus.antenna ? 2 : 1);
  Serial.print(F(" dst=")); Serial.print(lastStatus.dstState);
  Serial.print(F(" trk=")); Serial.println(lastStatus.tracking);

  if (ok) {
    // Per datasheet: time/status fields are only valid after a successful reception.
    ES100DateTime dt = es100.getDateTime();
    ES100NextDst  nd = es100.getNextDst();

    if (!esTimeSane(dt)) {
      // Do NOT poison the RTC with nonsense.
      ok = false;
      failCount++;
      es100.notifyRxResult(false);
      Serial.println(F("RX OK but time insane -> ignore"));
    } else {
      lastEsTime  = dt;
      lastNextDst = nd;

      okCount++;
      lastOkMs = millis();
      es100.notifyRxResult(true);

      // write RTC (LOCAL time)
      tmElements_t tm;
      tm.Year   = CalendarYrToTm(2000 + lastEsTime.year);
      tm.Month  = lastEsTime.month;
      tm.Day    = lastEsTime.day;
      tm.Hour   = lastEsTime.hour;
      tm.Minute = lastEsTime.minute;
      tm.Second = lastEsTime.second;

      RTC.write(tm);

      Serial.print(F("SET RTC: "));
      Serial.print(2000 + lastEsTime.year); Serial.print('-');
      Serial.print(lastEsTime.month); Serial.print('-');
      Serial.print(lastEsTime.day); Serial.print(' ');
      Serial.print(lastEsTime.hour); Serial.print(':');
      Serial.print(lastEsTime.minute); Serial.print(':');
      Serial.println(lastEsTime.second);
    }
  } else {
    failCount++;
    es100.notifyRxResult(false);
  }

  receiving = false;
  scheduleNext(ok);
}

static BtnEvent pollButton(BtnState& b) {
  // active-low with pullup
  bool rawHigh = (digitalRead(b.pin) == HIGH);
  unsigned long now = millis();

  // debounce: consider change stable if state persists for DEBOUNCE_MS
  static bool lastRawHigh[14]; // UNO pins 0..13
  static unsigned long lastRawChange[14];
  uint8_t p = b.pin;

  if (lastRawChange[p] == 0) { // init
    lastRawHigh[p] = rawHigh;
    lastRawChange[p] = now;
  }

  if (rawHigh != lastRawHigh[p]) {
    lastRawHigh[p] = rawHigh;
    lastRawChange[p] = now;
  }

  // stable?
  if ((now - lastRawChange[p]) >= DEBOUNCE_MS) {
    if (rawHigh != b.stableHigh) {
      // state changed
      b.stableHigh = rawHigh;

      if (!b.stableHigh) { // pressed
        b.pressedAtMs = now;
        b.longSent = false;
      } else { // released
        if (!b.longSent) return EV_SHORT;
      }
    }
  }

  if (!b.stableHigh && !b.longSent && (now - b.pressedAtMs) >= LONGPRESS_MS) {
    b.longSent = true;
    return EV_LONG;
  }

  return EV_NONE;
}

static void lcdLine(uint8_t row, const char* s) {
  lcd.setCursor(0, row);
  uint8_t i = 0;
  for (; s[i] && i < 20; i++) lcd.print(s[i]);
  for (; i < 20; i++) lcd.print(' ');
}

static const char* antModeStr(uint8_t m) {
  switch (m) {
    case ES100_ANTMODE_AUTO: return "auto";
    case ES100_ANTMODE_ANT1: return "ant1";
    case ES100_ANTMODE_ANT2: return "ant2";
    case ES100_ANTMODE_ALT_FAIL: return "alt";
  }
  return "?";
}


/*
  rxModeStr()
  -----------
  Convert the RX mode setting (stored as a small number in cfg.rxMode)
  into a short, user-friendly label for the LCD.

  cfg.rxMode values:
    0 -> "f"   : FULL reception only (most robust)
    1 -> "f+t" : FULL until first success, then TRACK
    2 -> "t"   : TRACK only (fast updates once signal is established)
*/
static const char* rxModeStr(uint8_t m) {
  switch (m) {
    case 0: return "f";
    case 1: return "f+t";
    case 2: return "t";
  }
  return "?";
}

/*
  dstModeStr()
  -----------
  DST mode label for the HOME/CLOCK screen.
*/
static const char* dstModeStr(uint8_t v) {
  switch (v) {
    case 0: return "off";  // DST disabled
    case 1: return "aut";  // DST automatic (current firmware uses this)
    case 2: return "on";   // reserved (manual DST on, if you ever add it)
  }
  return "?";
}


/*
  dstShortStr()
  ------------
  Short, fixed-width DST labels for the 20x4 LCD.

  The ES100 provides a 2-bit DST state (0..3). The long text variants like
  "ENDS TODAY" do not fit reliably on a 20-character line when combined with
  antenna + tracking info. These short labels are guaranteed to fit.

    NO  = no DST in effect
    END = DST ends today
    BEG = DST begins today
    ON  = DST currently active
*/
static const char* dstShortStr(uint8_t st) {
  switch (st & 0x03) {
    case 0: return "no";
    case 1: return "end";
    case 2: return "beg";
    case 3: return "yes";
  }
  return "?";
}

static const char* dstStateStr(uint8_t st) {
  switch (st & 0x03) {
    case 0: return "NO DST";
    case 1: return "ENDS TODAY";
    case 2: return "BEGINS TODAY";
    case 3: return "DST ACTIVE";
  }
  return "?";
}

static void fmtUptime(char* out, unsigned long ms) {
  unsigned long s = ms / 1000UL;
  unsigned long d = s / 86400UL; s %= 86400UL;
  unsigned long h = s / 3600UL;  s %= 3600UL;
  unsigned long m = s / 60UL;    s %= 60UL;

  if (d > 0) snprintf(out, 21, "%lud%luh", d, h);
  else if (h > 0) snprintf(out, 21, "%luh%lum", h, m);
  else if (m > 0) snprintf(out, 21, "%lum%lus", m, s);
  else snprintf(out, 21, "%lus", s);
}

// ===================== RENDERING =====================
/*
  LCD RENDERING
  -------------
  The UI is screen-based. S1/S3 cycle through screens.

  Screens included:
    1) CLOCK   : RTC local time + last sync age + OK/FAIL counters
    2) RX      : receiver state, last IRQ/status, antenna/tracking/DST state, last try age
    3) ES100   : ES100 device ID, last ES100 date/time, next DST info from the receiver
    4) ANALOG  : raw ADC readings from A0..A3 (0..1023)
    5) CONFIG  : editable settings stored in EEPROM

  Rendering is kept lightweight and purely "presentation": no hardware work is done here.
*/
static void renderClock() {
  tmElements_t tm;
  bool rtcOk = RTC.read(tm);

  char l0[21], l1[21], l2[21], l3[21];

  snprintf(l0, 21, "Z:%+d DST:%-3s RX:%-3s",
         (int)cfg.zoneHours,
         dstModeStr(cfg.dstAuto),
         rxModeStr(cfg.rxMode));

  if (rtcOk) {
    snprintf(l1, 21, "%04d-%02d-%02d %02d:%02d:%02d",
             tmYearToCalendar(tm.Year), tm.Month, tm.Day, tm.Hour, tm.Minute, tm.Second);
  } else {
    snprintf(l1, 21, "RTC READ FAILED     ");
  }

  char ago[21];
  if (lastOkMs == 0) snprintf(ago, 21, "never");
  else fmtUptime(ago, millis() - lastOkMs);

  snprintf(l2, 21, "LastOK:%-12s", ago);
  snprintf(l3, 21, "OK:%lu  FAIL:%lu", (unsigned long)okCount, (unsigned long)failCount);

  lcdLine(0, l0);
  lcdLine(1, l1);
  lcdLine(2, l2);
  lcdLine(3, l3);
}

static void renderRx() {
  char l0[21], l1[21], l2[21], l3[21];

  unsigned long dueS = 0;
  if (nextRxDueMs > millis()) dueS = (nextRxDueMs - millis()) / 1000UL;

  snprintf(l0, 21, "RX:%s Due:%lus",
           receiving ? "run " : "idle",
           (unsigned long)dueS);

  snprintf(l1, 21, "IRQ:0x%02X RXok:%s",
           lastIrqStatus,
           lastStatus.rxOk ? "yes" : "no");

  snprintf(l2, 21, "Ant:%d Trk:%c DST:%s",
           lastStatus.antenna ? 2 : 1,
           lastStatus.tracking ? 'Y' : 'N',
           dstShortStr(lastStatus.dstState));

  char ago[21];
  if (lastAttemptMs == 0) snprintf(ago, 21, "never");
  else fmtUptime(ago, millis() - lastAttemptMs);
  snprintf(l3, 21, "LastTry:%-10s", ago);

  lcdLine(0, l0);
  lcdLine(1, l1);
  lcdLine(2, l2);
  lcdLine(3, l3);
}

static void renderEs100() {
  char l0[21], l1[21], l2[21], l3[21];

  // If we never had a successful reception since boot, don't print misleading dates.
  if (okCount == 0) {
    snprintf(l0, 21, "ES100: waiting RX");
    snprintf(l1, 21, "IRQ:0x%02X  Antenna:%d", lastIrqStatus, lastStatus.antenna ? 2 : 1);
    snprintf(l2, 21, "DST:%s", dstStateStr(lastStatus.dstState));
    snprintf(l3, 21, "OK:%u  FAIL:%u", (unsigned)okCount, (unsigned)failCount);
  } else {
    snprintf(l0, 21, "ES100 IRQ:0x%02X", lastIrqStatus);

    snprintf(l1, 21, "ES DATE:%04d-%02d-%02d",
             2000 + lastEsTime.year, lastEsTime.month, lastEsTime.day);

    snprintf(l2, 21, "ES:%02d:%02d:%02d A:%d",
             lastEsTime.hour, lastEsTime.minute, lastEsTime.second,
             lastStatus.antenna ? 2 : 1);

    snprintf(l3, 21, "NEXT DST:%02d-%02d %02dh",
             lastNextDst.month, lastNextDst.day, lastNextDst.hour);
  }

  lcdLine(0, l0);
  lcdLine(1, l1);
  lcdLine(2, l2);
  lcdLine(3, l3);
}

static void renderAnalog() {
  char l1[21], l2[21], l3[21];
  int a0 = analogRead(A0);
  int a1 = analogRead(A1);
  int a2 = analogRead(A2);
  int a3 = analogRead(A3);

  lcdLine(0, "ANALOG INPUTS A0-A3 ");
  snprintf(l1, 21, "A0:%4d  A1:%4d", a0, a1);
  snprintf(l2, 21, "A2:%4d  A3:%4d", a2, a3);
  snprintf(l3, 21, "S2:menu  S2long:sync");

  lcdLine(1, l1);
  lcdLine(2, l2);
  lcdLine(3, l3);
}

static const uint8_t CFG_ITEMS = 6;
static const char* cfgItemName(uint8_t i) {
  switch (i) {
    case 0: return "ZONE";
    case 1: return "DST";
    case 2: return "ANT";
    case 3: return "SYNC";
    case 4: return "RETRY";
    case 5: return "RX";
  }
  return "?";
}

static void cfgAdjust(int dir) {
  switch (cfgIndex) {
    case 0: { // zone
      int z = (int)cfg.zoneHours + dir;
      if (z < -12) z = -12;
      if (z > 14)  z = 14;
      cfg.zoneHours = (int8_t)z;
    } break;
    case 1: { // dst auto toggle
      cfg.dstAuto = cfg.dstAuto ? 0 : 1;
    } break;
    case 2: { // ant mode cycle
      int m = (int)cfg.antMode + dir;
      if (m < ES100_ANTMODE_AUTO) m = ES100_ANTMODE_ALT_FAIL;
      if (m > ES100_ANTMODE_ALT_FAIL) m = ES100_ANTMODE_AUTO;
      cfg.antMode = (uint8_t)m;
    } break;
    case 3: { // ok minutes
      int v = (int)cfg.okMinutes + dir;
      if (v < 1) v = 1;
      if (v > 120) v = 120;
      cfg.okMinutes = (uint8_t)v;
    } break;
    case 4: { // fail minutes
      int v = (int)cfg.failMinutes + dir;
      if (v < 1) v = 1;
      if (v > 60) v = 60;
      cfg.failMinutes = (uint8_t)v;
    } break;
    case 5: { // rx mode
      int v = (int)cfg.rxMode + dir;
      if (v < 0) v = 2;
      if (v > 2) v = 0;
      cfg.rxMode = (uint8_t)v;
    } break;
  }
  cfgDirty = true;
  applyCfgToEs100();
}

static void renderCfg() {
  char l0[21], l1[21], l2[21], l3[21];

  snprintf(l0, 21, "CFG %s %c",
           cfgEditing ? "EDIT" : "NAV ",
           cfgDirty ? '*' : ' ');

  const char* name = cfgItemName(cfgIndex);

  char val[12];
  switch (cfgIndex) {
    case 0: snprintf(val, sizeof(val), "%+d", (int)cfg.zoneHours); break;
    case 1: snprintf(val, sizeof(val), "%s", cfg.dstAuto ? "auto" : "off"); break;
    case 2: snprintf(val, sizeof(val), "%s", antModeStr(cfg.antMode)); break;
    case 3: snprintf(val, sizeof(val), "%umin", (unsigned)cfg.okMinutes); break;
    case 4: snprintf(val, sizeof(val), "%umin", (unsigned)cfg.failMinutes); break;
    case 5:
      if (cfg.rxMode == 0) snprintf(val, sizeof(val), "full");
      else if (cfg.rxMode == 1) snprintf(val, sizeof(val), "full+track");
      else snprintf(val, sizeof(val), "track");
      break;
    default: snprintf(val, sizeof(val), "?"); break;
  }snprintf(l1, 21, ">%-5s:%-10s", name, val);
  snprintf(l2, 21, "S1/S3:%s S2:%s",
           cfgEditing ? "chg" : "nav",
           cfgEditing ? "done" : "edit");
  snprintf(l3, 21, "S2long:%s", cfgDirty ? "save+exit" : "exit");

  lcdLine(0, l0);
  lcdLine(1, l1);
  lcdLine(2, l2);
  lcdLine(3, l3);
}

static void renderScreen() {
  switch (g_screen) {
    case SCR_CLOCK:  renderClock(); break;
    case SCR_RX:     renderRx(); break;
    case SCR_ES100:  renderEs100(); break;
    case SCR_ANALOG: renderAnalog(); break;
    case SCR_CFG:    renderCfg(); break;
    default:         renderClock(); break;
  }
}

// ===================== SETUP / LOOP =====================
/*
  MAIN CONTROL FLOW
  -----------------
  setup():
    - initializes Serial, I²C, LCD, buttons, RTC, ES100
    - loads settings from EEPROM (or defaults)
    - starts the first reception attempt

  loop():
    - updates button state, handles navigation/menu logic
    - if an ES100 IRQ occurred: reads ES100 data, applies timezone/DST (library),
      updates the RTC, updates counters and schedules the next RX attempt
    - periodically refreshes the LCD screen
    - starts new RX attempts based on the scheduling rules and the chosen RX mode
*/
void setup() {
  Serial.begin(SERIAL_BAUD);

  Wire.begin();
  Wire.setClock(100000);

  pinMode(PIN_S1, INPUT_PULLUP);
  pinMode(PIN_S2, INPUT_PULLUP);
  pinMode(PIN_S3, INPUT_PULLUP);

  lcd.begin(20, 4);
  lcd.clear();
  lcdLine(0, "ES100 adk tech demo ");
  lcdLine(1, "s1/s3: screens       ");
  lcdLine(2, "S2 menu  S2L sync   ");
  lcdLine(3, "Boot...             ");

  cfgLoad(cfg);

  es100.begin(PIN_ES100_IRQ, PIN_ES100_EN);
  applyCfgToEs100();

  es100.enable();
  lastDeviceId = es100.getDeviceID();

  attachInterrupt(digitalPinToInterrupt(PIN_ES100_IRQ), isr_es100_irq, FALLING);

  // force an RX attempt soon after boot
  nextRxDueMs = 0;
  receiving = false;
  g_screen = SCR_CLOCK;

  Serial.print(F("ES100 DeviceID=0x"));
  Serial.println(lastDeviceId, HEX);
  Serial.print(F("CFG zone=")); Serial.print((int)cfg.zoneHours);
  Serial.print(F(" dstAuto=")); Serial.print((int)cfg.dstAuto);
  Serial.print(F(" antMode=")); Serial.print((int)cfg.antMode);
  Serial.print(F(" okMin=")); Serial.print((int)cfg.okMinutes);
  Serial.print(F(" failMin=")); Serial.print((int)cfg.failMinutes);
  Serial.print(F(" rxMode=")); Serial.println((int)cfg.rxMode);

  lcdLine(3, "waiting for rx...   ");
}

void loop() {
  unsigned long now = millis();

  // buttons
  BtnEvent e1 = pollButton(b1);
  BtnEvent e2 = pollButton(b2);
  BtnEvent e3 = pollButton(b3);

  if (g_screen != SCR_CFG) {
    if (e1 == EV_SHORT) g_screen = (Screen)((g_screen == 0) ? (SCR_COUNT - 1) : (g_screen - 1));
    if (e3 == EV_SHORT) g_screen = (Screen)((g_screen + 1) % SCR_COUNT);

    if (e2 == EV_SHORT) {
      g_screen = SCR_CFG;
      cfgEditing = false;
      cfgIndex = 0;
    }

    if (e2 == EV_LONG) {
      // force RX now
      nextRxDueMs = 0;
    }
  } else {
    // config menu
    if (e1 == EV_SHORT) {
      if (cfgEditing) cfgAdjust(+1);
      else cfgIndex = (cfgIndex == 0) ? (CFG_ITEMS - 1) : (cfgIndex - 1);
    }
    if (e3 == EV_SHORT) {
      if (cfgEditing) cfgAdjust(-1);
      else cfgIndex = (uint8_t)((cfgIndex + 1) % CFG_ITEMS);
    }
    if (e2 == EV_SHORT) {
      cfgEditing = !cfgEditing;
    }
    if (e2 == EV_LONG) {
      if (cfgDirty) cfgSave(cfg);
      cfgDirty = false;
      cfgEditing = false;
      g_screen = SCR_CLOCK;
    }
  }

  // RX scheduling
  if (!receiving && (long)(now - nextRxDueMs) >= 0) {
    bool tracking = false;
    if (cfg.rxMode == 2) tracking = true;
    else if (cfg.rxMode == 1) tracking = (okCount > 0);

    startReception(tracking);

    // prevent immediate re-trigger if ES100 is slow to IRQ
    nextRxDueMs = now + 60000UL;
  }

  // handle IRQ results
  handleIrqIfAny();

  // UI refresh
  if ((now - g_lastUiMs) >= UI_REFRESH_MS) {
    renderScreen();
    g_lastUiMs = now;
  }
}
